import os
import unittest
import tempfile
import math
from unittest.mock import patch, MagicMock

# Import the classes we want to test
from file_manager.xfer_timing import CSVData, TransferTiming


class TestCSVData(unittest.TestCase):
    """Test cases for the CSVData class"""

    def setUp(self):
        """Set up test fixtures"""
        # Create a temporary CSV file for testing
        self.temp_file = tempfile.NamedTemporaryFile(mode='w+', delete=False, suffix='.csv')
        self.temp_file.write("From,System1,System2,System3\n")
        self.temp_file.write("System1,0,10.5,20.3\n")
        self.temp_file.write("System2,15.2,0,25.7\n")
        self.temp_file.write("System3,18.9,22.4,0\n")
        self.temp_file.close()

        # Create a CSVData instance for testing
        self.csv_data = CSVData(self.temp_file.name, index_col=0)

    def tearDown(self):
        """Tear down test fixtures"""
        os.unlink(self.temp_file.name)

    def test_read_csv(self):
        """Test that the CSV file is read correctly"""
        # Check columns
        self.assertEqual(self.csv_data.columns, ['System1', 'System2', 'System3'])

        # Check index
        self.assertEqual(self.csv_data.index, ['System1', 'System2', 'System3'])

        # Check data
        self.assertEqual(self.csv_data.data['System1']['System2'], 10.5)
        self.assertEqual(self.csv_data.data['System2']['System3'], 25.7)
        self.assertEqual(self.csv_data.data['System3']['System1'], 18.9)

    def test_at_method(self):
        """Test the at method for accessing values"""
        # Test valid access
        self.assertEqual(self.csv_data.at('System1', 'System2'), 10.5)
        self.assertEqual(self.csv_data.at('System2', 'System3'), 25.7)

        # Test invalid access
        self.assertTrue(math.isnan(self.csv_data.at('System1', 'NonExistentSystem')))
        self.assertTrue(math.isnan(self.csv_data.at('NonExistentSystem', 'System1')))

    def test_drop_method(self):
        """Test the drop method for removing columns"""
        # Drop a column
        result = self.csv_data.drop(['System2'])

        # Check that the original data is unchanged
        self.assertEqual(self.csv_data.columns, ['System1', 'System2', 'System3'])

        # Check that the result has the correct columns
        self.assertEqual(result.columns, ['System1', 'System3'])

        # Check that the data is correct
        self.assertEqual(result.data['System1']['System3'], 20.3)
        self.assertEqual(result.data['System3']['System1'], 18.9)

        # Check that the dropped column is not in the result
        self.assertNotIn('System2', result.columns)
        self.assertNotIn('System2', result.data['System1'])

    def test_idxmin_method(self):
        """Test the idxmin method for finding the minimum value"""
        # Test with all values
        # System2 has the lowest non-zero value in the System1 column (15.2)
        self.assertEqual(self.csv_data.idxmin('System1'), 'System2')
        # System1 has the lowest non-zero value in the System2 column (10.5)
        self.assertEqual(self.csv_data.idxmin('System2'), 'System1')

        # Test with exclude_zeros=False
        # System1 has the lowest value (0) in the System1 column when including zeros
        self.assertEqual(self.csv_data.idxmin('System1', exclude_zeros=False), 'System1')

        # Test with a column that doesn't exist
        self.assertIsNone(self.csv_data.idxmin('NonExistentSystem'))


@patch('file_manager.xfer_timing.ilogger.addFileLogger')
@patch('file_manager.xfer_timing.speedtest')
class TestTransferTiming(unittest.TestCase):
    """Test cases for the TransferTiming class"""

    def setUp(self):
        """Set up test fixtures"""
        # Create a temporary CSV file for testing
        self.temp_file = tempfile.NamedTemporaryFile(mode='w+', delete=False, suffix='.csv')
        self.temp_file.write("From,System1,System2,System3\n")
        self.temp_file.write("System1,0,10.5,20.3\n")
        self.temp_file.write("System2,15.2,0,25.7\n")
        self.temp_file.write("System3,18.9,22.4,0\n")
        self.temp_file.close()

        # Create a mock logger
        self.mock_logger = MagicMock()

    def tearDown(self):
        """Tear down test fixtures"""
        os.unlink(self.temp_file.name)

    # Helper method to configure the mocks
    def _configure_mocks(self, mock_speedtest, mock_logger):
        # Configure the logger mock
        mock_logger.return_value = self.mock_logger

        # Configure the speedtest mock
        """Configure the speedtest mock to return predetermined values"""
        # Create a mock Speedtest instance
        mock_speedtest_instance = MagicMock()

        # Configure the mock to return our test values
        mock_speedtest_instance.download.return_value = 2000000  # 2 Mbps
        mock_speedtest_instance.upload.return_value = 1000000  # 1 Mbps

        # Configure the Speedtest class to return our mock instance
        mock_speedtest.Speedtest.return_value = mock_speedtest_instance

    def test_get_download_mbs(self, mock_speedtest, mock_logger):
        """Test getting download speed in Mbps"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        self.assertEqual(transfer_timing.get_download_mbs(), 2.0)

    def test_get_upload_mbs(self, mock_speedtest, mock_logger):
        """Test getting upload speed in Mbps"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        self.assertEqual(transfer_timing.get_upload_mbs(), 1.0)

    def test_get_upload_bps(self, mock_speedtest, mock_logger):
        """Test getting upload speed in bps"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        self.assertEqual(transfer_timing.get_upload_bps(), 1000000)

    @patch('os.stat')
    def test_get_local_speed_upload_time_estimate(self, mock_stat, mock_speedtest, mock_logger):
        """Test estimating upload time for a local file"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        # Mock the os.stat result
        mock_stat_result = MagicMock()
        mock_stat_result.st_size = 125000  # 125 KB
        mock_stat.return_value = mock_stat_result

        # Calculate expected result: (8 bits/byte * 125000 bytes) / 1000000 bits/sec = 1 second
        expected_time = 1.0

        # Test the method
        result = transfer_timing.get_local_speed_upload_time_estimate('/path/to/file')
        self.assertEqual(result, expected_time)

        # Test with a non-existent file
        mock_stat.side_effect = OSError()
        result = transfer_timing.get_local_speed_upload_time_estimate('/non/existent/file')
        self.assertEqual(result, 99999999999)

    def test_get_transfer_speed_in_mbs(self, mock_speedtest, mock_logger):
        """Test getting transfer speed between systems"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        # Test valid systems
        self.assertEqual(transfer_timing.get_transfer_speed_in_mbs('System1', 'System2'), 10.5)
        self.assertEqual(transfer_timing.get_transfer_speed_in_mbs('System2', 'System3'), 25.7)

        # Test invalid systems
        result = transfer_timing.get_transfer_speed_in_mbs('System1', 'NonExistentSystem')
        self.assertTrue(math.isnan(result))

        result = transfer_timing.get_transfer_speed_in_mbs('NonExistentSystem', 'System1')
        self.assertTrue(math.isnan(result))

    def test_get_best_system_basic(self, mock_speedtest, mock_logger):
        """Test finding the best system for transfers - basic case"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        # Test without system limits
        result = transfer_timing.get_best_system('System1')
        self.assertEqual(result, 'System2')

    def test_get_best_system_reverse(self, mock_speedtest, mock_logger):
        """Test finding the best system for transfers - reverse case"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        # Test the reverse direction
        result = transfer_timing.get_best_system('System2')
        self.assertEqual(result, 'System1')

    def test_get_best_system_with_limits(self, mock_speedtest, mock_logger):
        """Test finding the best system for transfers - with system limits"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        # Test with system limits
        result = transfer_timing.get_best_system('System1', systems_limits=['System2', 'System3'])
        self.assertEqual(result, 'System2')

    def test_get_best_system_invalid(self, mock_speedtest, mock_logger):
        """Test finding the best system for transfers - invalid system"""
        # Configure the mocks
        self._configure_mocks(mock_speedtest, mock_logger)

        # Create the TransferTiming instance after configuring the mock
        transfer_timing = TransferTiming(file_transfer_path=self.temp_file.name)

        # Test with invalid system
        result = transfer_timing.get_best_system('NonExistentSystem')
        self.assertIsNone(result)


if __name__ == '__main__':
    unittest.main()
