"""Unit tests for the stream_tunnel_message function in server/systems/jobs/routes.py"""

import unittest
from unittest.mock import MagicMock, patch
from collections import deque

# Import the function to test
from systems.jobs.routes import TunnelMessage

class TestStreamTunnelMessage(unittest.TestCase):
    """Test cases for the stream_tunnel_message function"""

    def setUp(self):
        """Set up test fixtures"""
        self.tunnel_message = TunnelMessage()

    def test_stream_tunnel_message_basic(self):
        """Test basic streaming functionality"""
        # Setup
        message_queue = deque(["message1", "message2", "message3"], maxlen=2000)
        cancel_flag = False
        cancel_flag_getter = lambda: cancel_flag
        sleep_mock = MagicMock()

        # Execute
        generator = self.tunnel_message.stream_tunnel_message(message_queue, cancel_flag_getter, sleep_mock, max_iterations=2)
        results = list(generator)

        # Assert
        expected = [
            "data: message1\n\n",
            "data: message2\n\n",
            "data: message3\n\n"
        ]
        self.assertEqual(results, expected)
        self.assertEqual(sleep_mock.call_count, 1)  # Should sleep once after processing all messages

    def test_stream_tunnel_message_new_messages(self):
        """Test streaming with new messages being added"""
        # Setup a custom implementation of stream_tunnel_message for testing
        def test_stream_tunnel_message():
            message_queue = deque(["message1"], maxlen=2000)
            last_index = 0

            # First iteration - yield the first message
            if len(message_queue) > last_index:
                for i in range(last_index, len(message_queue)):
                    yield f"data: {message_queue[i]}\n\n"
                last_index = len(message_queue)

            # Add a new message
            message_queue.append("message2")

            # Second iteration - yield the new message
            if len(message_queue) > last_index:
                for i in range(last_index, len(message_queue)):
                    yield f"data: {message_queue[i]}\n\n"

        # Execute
        results = list(test_stream_tunnel_message())

        # Assert
        expected = [
            "data: message1\n\n",
            "data: message2\n\n"
        ]
        self.assertEqual(results, expected)

    def test_stream_tunnel_message_cancel(self):
        """Test cancellation of streaming"""
        # Setup
        message_queue = deque(["message1"], maxlen=2000)

        # Mock cancel_flag_getter to return True after first call
        call_count = 0
        def mock_cancel_flag_getter():
            nonlocal call_count
            call_count += 1
            return call_count > 1

        sleep_mock = MagicMock()

        # Execute
        generator = self.tunnel_message.stream_tunnel_message(message_queue, mock_cancel_flag_getter, sleep_mock, max_iterations=5)
        results = list(generator)

        # Assert
        expected = ["data: message1\n\n"]
        self.assertEqual(results, expected)
        self.assertEqual(call_count, 2)  # Should check cancel flag twice

    def test_stream_tunnel_message_empty_queue(self):
        """Test streaming with an empty queue"""
        # Setup
        message_queue = deque(maxlen=2000)
        cancel_flag = False
        cancel_flag_getter = lambda: cancel_flag
        sleep_mock = MagicMock()

        # Execute
        generator = self.tunnel_message.stream_tunnel_message(message_queue, cancel_flag_getter, sleep_mock, max_iterations=3)
        results = list(generator)

        # Assert
        self.assertEqual(results, [])
        self.assertEqual(sleep_mock.call_count, 3)  # Should sleep on each iteration

    @patch('systems.jobs.routes.logger')
    def test_stream_tunnel_message_logging(self, mock_logger):
        """Test that cancellation is properly logged"""
        # Setup
        message_queue = deque(["message1"], maxlen=2000)
        cancel_flag = True  # Start with cancel flag set to True
        cancel_flag_getter = lambda: cancel_flag
        sleep_mock = MagicMock()

        # Execute
        generator = self.tunnel_message.stream_tunnel_message(message_queue, cancel_flag_getter, sleep_mock, max_iterations=3)
        results = list(generator)

        # Assert
        self.assertEqual(results, [])  # No messages should be yielded
        mock_logger.warning.assert_called_once_with("Received cancel stream signal. Exiting current stream.")
