"""Tests for systems.json files
"""

import unittest
import json
import os.path

# 3rd Party imports
from jsonschema import validate
from plugins.filters import is_safe

class TestSystemSyntax(unittest.TestCase):
    """Validates all plugins in `api/plugins/` using our JSON Schema.
    """
    # Get file paths relative to this file's known location
    system_schema_path = os.path.join(os.path.dirname(__file__), "../schemas/system-schema.json")
    features_schema_path = os.path.join(os.path.dirname(__file__), "../schemas/features-schema.json")
    systems_file = os.path.join(os.path.dirname(__file__), "../systems.json")
    features_dir = os.path.join(os.path.dirname(__file__), "../smartqueue/features")
    # Class variable holding the current JSON Schema. See setUpClass() for more info
    system_schema = None
    features_schema = None

    @classmethod
    def setUpClass(cls):
        """Open and read in the current system and features JSON Schemas from disk for use during testing.
        """
        # Load system schema
        with open(cls.system_schema_path, mode='r') as f:
            cls.system_schema = json.load(f)

        # Load features schema
        with open(cls.features_schema_path, mode='r') as f:
            cls.features_schema = json.load(f)

        cls.errors = []

    @classmethod
    def tearDownClass(cls):
        """Run once after all tests in the class have been run.
        Report all accumulated errors here.
        """
        if len(cls.errors) > 0:
            print("\nErrors found during system validation:")
            for err in cls.errors:
                print("-"*70)
                try:
                    msg = err["error"].message
                except AttributeError:
                    msg = err["error"]
                print(f"Error validating system '{err['system']}': {msg} -- {err['file']}")
            # This assertion will make the test suite fail if any errors were found
            assert False, f"{len(cls.errors)} errors found during system validation"

    def _test_system_data(self, system, data, file):
        """Runs a single test against the provided system's data

        Args:
            system (str): Name of the system for the provided data
            data (dict): The systems data to test
            file (str): The name of the file data belongs to
        """
        try:
            self.assertIsNone(validate(instance=data, schema=self.system_schema))
        except Exception as e:
            self.errors.append({"system": system, "file": os.path.basename(file), "error": e})

    def test_isvalid(self):
        """For each systems file, validate each system in file against our schema. If an
        exception is raised, the plugin file is NOT valid.
        """
        with open(self.systems_file, mode='r') as f:
            systems = json.load(f).get("systems")
            if systems:
                for system, system_data in systems.items():
                    self._test_system_data(system, system_data, self.systems_file)
            else:
                self.errors[self.systems_file] = "Has no systems attribute."

    def test_issafe(self):
        """Ensure each system name and node type passes the is_safe constraint provided by plugins.filters"""
        with open(self.systems_file, mode='r') as f:
            systems = json.load(f).get("systems")
            if systems:
                for system, system_data in systems.items():
                    if not is_safe(system):
                        self.errors.append({"system": system, "file": os.path.basename(self.systems_file), "error": "System name fails 'is_safe' constraint"})
                    for node in system_data.get("nodes", []):
                        node_name = node.get("name")
                        if not is_safe(node_name):
                            self.errors.append({"system": system, "file": os.path.basename(self.systems_file), "error": f"Node name '{node_name}' fails 'is_safe' constraint"})

    def _validate_feature_file(self, system, feature_file_path):
        """Validates a feature file against the features schema.

        Args:
            system (str): Name of the system the feature file belongs to
            feature_file_path (str): Path to the feature file to validate
        """
        try:
            # Check if file exists (though this should be handled by test_feature_files)
            if not os.path.isfile(feature_file_path):
                self.errors.append({
                    "system": system,
                    "file": feature_file_path,
                    "error": "Feature file does not exist"
                })
                return

            # Load and validate the feature file
            with open(feature_file_path, mode='r') as f:
                try:
                    feature_data = json.load(f)
                except json.JSONDecodeError as e:
                    self.errors.append({
                        "system": system,
                        "file": os.path.basename(feature_file_path),
                        "error": f"Invalid JSON: {str(e)}"
                    })
                    return

            # Validate against the features schema
            validate(instance=feature_data, schema=self.features_schema)

            # Return the feature data for further validation if needed
            return feature_data

        except Exception as e:
            self.errors.append({
                "system": system,
                "file": os.path.basename(feature_file_path),
                "error": e
            })
            return None

    def test_feature_files(self):
        """Ensures each active system has an associated features file with expected schema
        and property value matching between systems.json and feature files.
        """
        with open(self.systems_file, mode='r') as f:
            systems = json.load(f).get("systems")

        for system, system_data in systems.items():
            if not system_data.get("active", False):
                continue

            features = system_data.get("provides_features")
            # Property must exist with a value
            if not features:
                self.errors.append({
                    "system": system,
                    "file": os.path.basename(self.systems_file),
                    "error": "Does not define a features file with the 'provides_features' property"
                })
                continue

            # Construct the path to the features file
            feature_file_path = os.path.join(self.features_dir, features)

            # Validate the features file against the schema
            feature_data = self._validate_feature_file(system, feature_file_path)
            if not feature_data:
                # Error already logged in _validate_feature_file
                continue

            # Check that node names in systems.json align with node names in features file
            system_node_names = {node.get("name") for node in system_data.get("nodes", [])}
            feature_node_names = set()

            # Extract node names from feature data
            # The structure might vary based on your schema, adjust as needed
            for node in feature_data.get("nodes", []):
                feature_node_names.add(node["typeConfig"]["ilauncher"])

            # Check for nodes in systems.json not present in features file
            missing_in_features = system_node_names - feature_node_names
            if missing_in_features:
                self.errors.append({
                    "system": system,
                    "file": os.path.basename(feature_file_path),
                    "error": f"Nodes found in systems.json but missing in features file: {', '.join(missing_in_features)}"
                })
            # Check for nodes in features file not present in systems.json
            missing_in_system = feature_node_names - system_node_names
            if missing_in_system:
                self.errors.append({
                    "system": system,
                    "file": os.path.basename(feature_file_path),
                    "error": f"Nodes found in features file but missing in systems.json: {', '.join(missing_in_system)}"
                })


