import os
import json
import unittest
from unittest.mock import patch, MagicMock

# Import the modules we're testing
from smartqueue.feature_utils import (
    load_feature_files,
    get_filter_options,
    filter_systems_by_criteria,
    node_map,
    get_num_cores,
    embed_features_in_systems
)

class TestSystemFeatures(unittest.TestCase):
    # Get file paths relative to this file's known location
    systems_file = os.path.join(os.path.dirname(__file__), "../systems.json")
    features_dir = os.path.join(os.path.dirname(__file__), "../smartqueue/features")
    # class level data
    systems = None
    features = []
    processed_features = []

    @classmethod
    def setUpClass(cls):
        """Open and read in the current system and features JSON data from disk for use during testing.
        """
        # Load systems data
        with open(cls.systems_file, mode='r') as f:
            cls.systems = json.load(f)

        # Load features data - corrected to handle a directory of files
        cls.features = []
        cls.processed_features = []

        for filename in os.listdir(cls.features_dir):
            if filename.endswith('.json'):
                file_path = os.path.join(cls.features_dir, filename)
                with open(file_path, mode='r') as f:
                    feature_data = json.load(f)
                    cls.features.append(feature_data)

                    # Process the feature data similar to how our implementation will
                    system_name = feature_data.get('system')
                    architecture = feature_data.get('architecture')

                    for node in feature_data.get('nodes', []):
                        node_type = node.get('name')
                        ilauncher_type = node.get('typeConfig', {}).get('ilauncher')

                        node_data = {
                            'System': system_name,
                            'Architecture': architecture,
                            'Node_Type': node_type,
                            'ILauncher_Type': ilauncher_type,
                            'CPU_RAM': node.get('ram', 0),
                            'CPU_Cores': node.get('cpu', {}).get('cores', 0),
                            'GPU_Count': node.get('gpu', {}).get('count', 0),
                            'GPU_RAM': node.get('gpu', {}).get('ram', 0),
                            'CUDA_Version': node.get('cuda', {}).get('versions', []),
                            'Node_Types': node.get('typeConfig', {})
                        }
                        cls.processed_features.append(node_data)

    def test_load_feature_files(self):
        """Test that feature files are loaded correctly."""
        # Load feature files using our implementation
        loaded_data = load_feature_files(self.features_dir)

        # Check that we have data
        self.assertTrue(len(loaded_data) > 0, "No feature data was loaded")

        # Check that all systems are represented
        systems_in_loaded = {item['System'] for item in loaded_data}
        systems_in_reference = {item['System'] for item in self.processed_features}
        self.assertEqual(systems_in_loaded, systems_in_reference,
                         "Loaded systems don't match reference systems")

        # Check that the data structure is correct
        for item in loaded_data:
            self.assertIn('System', item)
            self.assertIn('Architecture', item)
            self.assertIn('Node_Type', item)
            self.assertIn('CPU_RAM', item)
            self.assertIn('CPU_Cores', item)
            self.assertIn('GPU_Count', item)
            self.assertIn('GPU_RAM', item)
            self.assertIn('CUDA_Version', item)

    def test_embed_features_in_systems(self):
        """Test embedding feature data in systems.json."""
        # Create a copy of systems data for testing
        import tempfile
        temp_path = tempfile.mktemp(suffix='.json')
        with open(temp_path, 'w') as temp_file:
            # Write systems data to temp file
            json.dump(self.systems, temp_file)

        try:
            # Call the function to embed features
            updated_systems = embed_features_in_systems(temp_path, self.features_dir, output_path=None)

            # Check that active systems have features embedded
            active_systems_with_features = 0
            for system_name, system_info in updated_systems['systems'].items():
                if system_info.get('active', False):
                    if system_info.get('provides_features'):
                        self.assertIn('features', system_info,
                                     f"Active system {system_name} with provides_features doesn't have features embedded")
                        active_systems_with_features += 1

            # Ensure we processed at least one system
            self.assertTrue(active_systems_with_features > 0,
                           "No active systems had features embedded")
        finally:
            # Clean up temp file
            if os.path.exists(temp_path):
                os.unlink(temp_path)

    def test_get_filter_options(self):
        """Test extracting filter options from feature data."""
        # Get filter options
        filter_options = get_filter_options(self.processed_features)

        # Check architecture options
        self.assertIn('arch', filter_options, "Filter options missing 'arch' key")
        self.assertIn('All', filter_options['arch'], "Architecture options missing 'All' option")

        # Check that all architectures from processed features are included
        architectures = set(item['Architecture'] for item in self.processed_features)
        for arch in architectures:
            self.assertIn(arch, filter_options['arch'],
                         f"Architecture '{arch}' missing from filter options")

        # Check CUDA version options
        self.assertIn('cudaVersion', filter_options, "Filter options missing 'cudaVersion' key")
        self.assertIn(None, filter_options['cudaVersion'], "CUDA version options missing None option")

        # Check that all CUDA versions from processed features are included
        cuda_versions = set()
        for item in self.processed_features:
            cuda_versions.update(item.get('CUDA_Version', []))

        for version in cuda_versions:
            self.assertIn(version, filter_options['cudaVersion'],
                         f"CUDA version '{version}' missing from filter options")

    def test_filter_systems_by_criteria(self):
        """Test filtering systems based on user criteria."""
        # Define test cases for various filtering scenarios
        test_cases = [
            {
                "name": "Filter by CPU RAM (>=)",
                "criteria": {
                    "ram": {"value": 500, "operator": ">="},
                    "cpu": {"value": None},
                    "gpu": {"value": None},
                    "gpu_ram": {"value": None},
                    "arch": {"value": None},
                    "cudaVersion": {"value": None, "operator": ">="},
                    "systems": {"value": []},
                    "accounts": {"value": []}
                }
            },
            {
                "name": "Filter by CPU cores (==)",
                "criteria": {
                    "ram": {"value": None},
                    "cpu": {"value": 128, "operator": "=="},
                    "gpu": {"value": None},
                    "gpu_ram": {"value": None},
                    "arch": {"value": None},
                    "cudaVersion": {"value": None, "operator": ">="},
                    "systems": {"value": []},
                    "accounts": {"value": []}
                }
            },
            {
                "name": "Filter by GPU count (>=)",
                "criteria": {
                    "ram": {"value": None},
                    "cpu": {"value": None},
                    "gpu": {"value": 2, "operator": ">="},
                    "gpu_ram": {"value": None},
                    "arch": {"value": None},
                    "cudaVersion": {"value": None, "operator": ">="},
                    "systems": {"value": []},
                    "accounts": {"value": []}
                }
            },
            {
                "name": "Filter by architecture",
                "criteria": {
                    "ram": {"value": None},
                    "cpu": {"value": None},
                    "gpu": {"value": None},
                    "gpu_ram": {"value": None},
                    "arch": {"value": "x86-64"},
                    "cudaVersion": {"value": None, "operator": ">="},
                    "systems": {"value": []},
                    "accounts": {"value": []}
                }
            }
        ]

        # Run tests for each test case
        for test_case in test_cases:
            with self.subTest(test_case["name"]):
                result = filter_systems_by_criteria(self.processed_features, test_case["criteria"])

                # Check that result is a dictionary
                self.assertIsInstance(result, dict, f"Result for {test_case['name']} is not a dictionary")

                # For each system in the result, check that it has a list of node types
                for system, node_types in result.items():
                    self.assertIsInstance(node_types, list,
                                         f"Node types for system {system} is not a list")

    def test_node_map(self):
        """Test node mapping functionality."""
        # Get node maps
        node_maps = node_map(self.processed_features)

        # Check that node_maps is a dictionary
        self.assertIsInstance(node_maps, dict, "Node map is not a dictionary")

        # Check that all systems have node maps
        systems_in_features = {item['System'].lower() for item in self.processed_features}
        for system in systems_in_features:
            self.assertIn(system, node_maps, f"System {system} missing from node maps")

        # Check structure of node maps
        for system, nodes in node_maps.items():
            self.assertIsInstance(nodes, dict, f"Nodes for system {system} is not a dictionary")

            # Check at least one node has expected structure
            if nodes:
                for node_type, config in nodes.items():
                    self.assertIsInstance(config, dict,
                                         f"Config for node {node_type} in system {system} is not a dictionary")

    def test_get_num_cores(self):
        """Test getting number of cores for a system and node type."""
        # Find a valid system and node type from processed features
        test_system = None
        test_node = None
        expected_cores = None

        for item in self.processed_features:
            if item['System'] == 'Narwhal' and item['Node_Type'] == 'Login':
                test_system = item['System']
                test_node = item['Node_Type']
                expected_cores = item['CPU_Cores']
                break

        # If we found a test case, run the test
        if test_system and test_node:
            # Test valid system and node type
            cores = get_num_cores(self.processed_features, test_system, test_node)
            self.assertEqual(cores, expected_cores,
                            f"Incorrect core count for {test_system}:{test_node}")

        # Test invalid system
        cores = get_num_cores(self.processed_features, "NonExistentSystem", "Login")
        self.assertIsNone(cores, "Non-existent system should return None")

        # Test invalid node type
        cores = get_num_cores(self.processed_features, "Narwhal", "NonExistentNode")
        self.assertIsNone(cores, "Non-existent node type should return None")
