"""
Simple unit tests for SSE (Server-Sent Events) broadcasting functionality.
Tests the ClientManager and event broadcasting to multiple browser tabs.
"""

import unittest
import queue
import time
import threading
from systems.events import ClientManager, ServerSentEvents


class TestClientManager(unittest.TestCase):
    """Simple tests for the ClientManager class"""

    def setUp(self):
        """Create a fresh ClientManager for each test"""
        self.manager = ClientManager()

    def test_register_single_client(self):
        """Test that we can register a client and get back a queue"""
        # Register a client
        client_id = "test-client-123"
        client_queue = self.manager.register_client(client_id)

        # Check we got valid returns
        self.assertIsInstance(client_queue, queue.Queue)

        # Check client count
        self.assertEqual(self.manager.get_client_count(), 1)
        self.assertEqual(self.manager.get_total_connection_count(), 1)

    def test_multiple_connections_same_client(self):
        """Test that same client can have multiple connections"""
        client_id = "test-client-456"

        # Register same client twice (simulating two connections)
        queue1 = self.manager.register_client(client_id)
        queue2 = self.manager.register_client(client_id)

        # Should be the same queue
        self.assertIs(queue1, queue2)

        # Should have 1 client with 2 connections
        self.assertEqual(self.manager.get_client_count(), 1)
        self.assertEqual(self.manager.get_total_connection_count(), 2)

    def test_unregister_client_with_multiple_connections(self):
        """Test that client is only removed when all connections are closed"""
        client_id = "test-client-789"

        # Register same client 3 times
        self.manager.register_client(client_id)
        self.manager.register_client(client_id)
        self.manager.register_client(client_id)

        self.assertEqual(self.manager.get_total_connection_count(), 3)

        # Unregister once - client should still exist
        self.manager.unregister_client(client_id)
        self.assertEqual(self.manager.get_client_count(), 1)
        self.assertEqual(self.manager.get_total_connection_count(), 2)

        # Unregister again - client should still exist
        self.manager.unregister_client(client_id)
        self.assertEqual(self.manager.get_client_count(), 1)
        self.assertEqual(self.manager.get_total_connection_count(), 1)

        # Unregister last time - client should be removed
        self.manager.unregister_client(client_id)
        self.assertEqual(self.manager.get_client_count(), 0)
        self.assertEqual(self.manager.get_total_connection_count(), 0)

    def test_broadcast_to_multiple_clients(self):
        """Test that broadcast sends event to all clients"""
        # Register 3 different clients
        client1_queue = self.manager.register_client("client-1")
        client2_queue = self.manager.register_client("client-2")
        client3_queue = self.manager.register_client("client-3")

        # Broadcast a test event
        test_event = {"event": "test", "data": "hello"}
        self.manager.put_event(test_event)

        # Each queue should have the event
        # Use timeout to avoid hanging if something goes wrong
        event1 = client1_queue.get(timeout=1)
        event2 = client2_queue.get(timeout=1)
        event3 = client3_queue.get(timeout=1)

        # All should have received the same event
        self.assertEqual(event1, test_event)
        self.assertEqual(event2, test_event)
        self.assertEqual(event3, test_event)

    def test_broadcast_to_empty_clients(self):
        """Test that broadcasting with no clients doesn't crash"""
        # No clients registered
        self.assertEqual(self.manager.get_client_count(), 0)

        # This should not raise an exception
        test_event = {"event": "test", "data": "hello"}
        self.manager.put_event(test_event)

        # Still no clients
        self.assertEqual(self.manager.get_client_count(), 0)

    def test_different_clients_get_different_queues(self):
        """Test that different clients get different queues"""
        # Register different clients
        queue1 = self.manager.register_client("client-a")
        queue2 = self.manager.register_client("client-b")
        queue3 = self.manager.register_client("client-c")

        # All queues should be different
        self.assertIsNot(queue1, queue2)
        self.assertIsNot(queue2, queue3)
        self.assertIsNot(queue1, queue3)

        # Should have 3 clients
        self.assertEqual(self.manager.get_client_count(), 3)


class TestServerSentEvents(unittest.TestCase):
    """Simple tests for ServerSentEvents integration with ClientManager"""

    def setUp(self):
        """Set up test with fresh SSE and ClientManager"""
        # Import the module to get fresh instances
        from systems import events

        # Create new instances for testing
        self.client_manager = ClientManager()
        self.sse = ServerSentEvents()

        # Replace the module's client_manager with our test one
        self.original_manager = events.client_manager
        events.client_manager = self.client_manager

    def tearDown(self):
        """Restore the original client_manager"""
        from systems import events
        events.client_manager = self.original_manager

    def test_sse_event_broadcasts_to_all_clients(self):
        """Test that SSE events are broadcast to all clients"""
        # Register 3 clients
        queue1 = self.client_manager.register_client("client-1")
        queue2 = self.client_manager.register_client("client-2")
        queue3 = self.client_manager.register_client("client-3")

        # Send a system status event through SSE
        self.sse.add_system_status_event("test-system", "up")

        # Check all queues received the event
        event1 = queue1.get(timeout=1)
        event2 = queue2.get(timeout=1)
        event3 = queue3.get(timeout=1)

        # Verify the event structure
        expected_event = {
            "event": "system_status",
            "system": "test-system",
            "data": {"status": "up"}
        }

        self.assertEqual(event1, expected_event)
        self.assertEqual(event2, expected_event)
        self.assertEqual(event3, expected_event)

    def test_different_event_types(self):
        """Test that different event types work correctly"""
        # Register a client
        client_queue = self.client_manager.register_client("test-client")

        # Test different event types
        self.sse.add_show_queues_event("system1", {"queue": "data"})
        self.sse.add_show_usage_event("system2", {"usage": "info"})
        self.sse.add_show_storage_event("system3", {"storage": "details"})

        # Get the events
        event1 = client_queue.get(timeout=1)
        event2 = client_queue.get(timeout=1)
        event3 = client_queue.get(timeout=1)

        # Check event types
        self.assertEqual(event1["event"], "show_queues")
        self.assertEqual(event2["event"], "show_usage")
        self.assertEqual(event3["event"], "show_storage")

    def test_client_with_multiple_connections_gets_single_event(self):
        """Test that client with multiple connections only gets one copy of each event"""
        client_id = "multi-connection-client"

        # Register same client multiple times (simulating multiple SSE connections)
        queue1 = self.client_manager.register_client(client_id)
        queue2 = self.client_manager.register_client(client_id)
        queue3 = self.client_manager.register_client(client_id)

        # All should be the same queue
        self.assertIs(queue1, queue2)
        self.assertIs(queue2, queue3)

        # Send an event
        self.sse.add_system_status_event("test-system", "up")

        # Only one event should be in the queue
        event = queue1.get(timeout=1)
        self.assertEqual(event["event"], "system_status")

        # Queue should now be empty
        self.assertTrue(queue1.empty())


class TestConcurrentClients(unittest.TestCase):
    """Test thread safety with concurrent clients"""

    def test_concurrent_client_registration(self):
        """Test that multiple threads can register clients safely"""
        manager = ClientManager()
        client_queues = []

        def register_client(client_id):
            """Helper function to register a client in a thread"""
            queue = manager.register_client(client_id)
            client_queues.append((client_id, queue))

        # Create 10 threads that register different clients
        threads = []
        for i in range(10):
            client_id = f"client-{i}"
            t = threading.Thread(target=register_client, args=(client_id,))
            threads.append(t)
            t.start()

        # Wait for all threads
        for t in threads:
            t.join()

        # Should have 10 clients
        self.assertEqual(manager.get_client_count(), 10)
        self.assertEqual(manager.get_total_connection_count(), 10)

        # All client IDs should be unique
        client_ids = [cid for cid, _ in client_queues]
        self.assertEqual(len(client_ids), len(set(client_ids)))

    def test_broadcast_during_client_registration(self):
        """Test broadcasting while clients are being added/removed"""
        manager = ClientManager()
        errors = []

        def register_and_listen(client_id):
            """Register, listen for events, then unregister"""
            try:
                client_queue = manager.register_client(client_id)
                # Try to get an event (might timeout if not broadcast yet)
                try:
                    event = client_queue.get(timeout=0.5)
                except queue.Empty:
                    pass  # It's ok if we don't get an event
                manager.unregister_client(client_id)
            except Exception as e:
                errors.append(e)

        def broadcast_events():
            """Broadcast events repeatedly"""
            for i in range(5):
                manager.put_event({"event": "test", "num": i})
                time.sleep(0.1)

        # Start broadcasting in a thread
        broadcast_thread = threading.Thread(target=broadcast_events)
        broadcast_thread.start()

        # Start multiple client threads
        client_threads = []
        for i in range(5):
            client_id = f"concurrent-client-{i}"
            t = threading.Thread(target=register_and_listen, args=(client_id,))
            client_threads.append(t)
            t.start()
            time.sleep(0.05)  # Stagger the connections slightly

        # Wait for all threads
        broadcast_thread.join()
        for t in client_threads:
            t.join()

        # Should have no errors
        self.assertEqual(len(errors), 0, f"Errors occurred: {errors}")

        # All clients should be cleaned up
        self.assertEqual(manager.get_client_count(), 0)


if __name__ == "__main__":
    # Run the tests with verbose output
    unittest.main(verbosity=2)