"""Tests for parsing each systems show_usage output.

To add a test create a file in show_usage dir with <system_name>.txt as the filename, and add an example output of this
system's show_usage command. The current show_usage parser expects a system name to look for within the column
"System" in the show_usage output. Please provide the appropriate system name as the filename to meet this expectation
for the tests.
"""

import os
import logging
import unittest
from engine import utils

class TestShowStorageParsing(unittest.TestCase):

    logger = logging.getLogger("show_usage_tests")
    data_dir = os.path.join(os.path.dirname(__file__), "show_usage")

    @classmethod
    def setUpClass(cls):
        """Open and read in the text from each system's show_usage text file.
        Note: Trim the whitespace at beginning of file to better simulate subprocess with shell=True.
        """
        cls.data = []
        for filename in os.listdir(cls.data_dir):
            if filename.endswith('.txt'):
                file_path = os.path.join(cls.data_dir, filename)
                system = os.path.splitext(filename)[0]
                with open(file_path, 'r') as file:
                    content = file.read()
                    cls.data.append({"system": system, "show_usage": content.lstrip()})
        cls.errors = []

    def tearDown(self):
        if len(self.errors) > 0:
            for err in self.errors:
                print("-"*70)
                print(f"Error parsing show_usage for '{err['system']}': {err['msg']}")
            self.assertIsNone(1)

    def test_parse_all_systems_show_usage_output(self):
        """Loops over all items read from show_usage/*.txt files and ensures the parser provides output.
        Also ensures the 'Subproject' key/value pair exists for each element of the parsed data.
        """
        for data in self.data:
            parsed_data = utils.parse_show_usage_stdout(data["show_usage"], data["system"], self.logger)
            # Ensure the system's show_usage output was parsed
            if not parsed_data:
                self.errors.append({"system": data["system"], "msg": "Func returned no data"})
                continue
            # Ensure we have at least a Subproject key value pair. The other column names (key names) can be arbitrary strings?
            for i, item in enumerate(parsed_data):
                if "Subproject" not in item.keys() or not item["Subproject"]:
                    self.errors.append({
                        "system": data["system"],
                        "msg": f"'Subproject' key/value pair not found for item '{i}'"
                    })


