"""Tests for parsing show_storage output and checking disk quota"""

import logging
import unittest
from engine import utils

class TestShowStorageParsing(unittest.TestCase):

    logger = logging.getLogger("show_storage_tests")

    def test_units_mismatch_header_and_value(self):
        stdout = """
        Navy DSRC User Disk Quota Information
                          As of 18:55 hours UTC on 27 Nov 2023

        System     Location                  Disk Usage (KB)  Disk Quota (KB)  File Usage  File Quota
        =========  ========================  ===============  ===============  ==========  ==========
        koehr      /p/home                          53898716        209715200       45878           0
        koehr      /p/work1                           893996      21474836480        5109           0
        koehr      /p/work2                                4                0           1           0
        koehr      /p/cwfs/rlarson                        0G          100000G           1           0
        """
        data = utils.parse_show_storage_stdout(stdout.strip(), self.logger)
        assert len(data) == 4 and data[0]["disk_usage_units"] == "KB" and data[3]["disk_quota_units"] == "G"

    def test_units_mismatch_usage_and_quota_with_quota_not_exceeded(self):
        stdout = """
                        MHPCC User Disk Storage Information
                        As of 0958 hours HST 08 November 2023

        System    Location    Disk Usage    Disk Quota    File Usage    File Quota
        ========  ==========  ============  ============  ============  ============
        Reef      LUSTRE        460.1G        1T         860858        1000000
        """
        data = utils.parse_show_storage_stdout(stdout.strip(), self.logger)
        assert data[0]["disk_quota_exceed"] == False

    def test_units_mismatch_usage_and_quota_with_quota_exceeded(self):
        # I haven't seen this one in the wild yet but who knows? (RKL)
        stdout = """
                        MHPCC User Disk Storage Information
                        As of 0958 hours HST 08 November 2023

        System    Location    Disk Usage    Disk Quota    File Usage    File Quota
        ========  ==========  ============  ============  ============  ============
        Reef      LUSTRE        4601G        1T         860858        1000000
        """
        data = utils.parse_show_storage_stdout(stdout.strip(), self.logger)
        assert data[0]["disk_quota_exceed"] == True

    def test_non_int_values(self):

        stdout = """
        ERDC DSRC User Disk Storage Information
                            As of 1255 hours CST 27 November 2023

        System      Location         Disk Usage     Disk Quota  File Usage  File Quota
        ==========  ============  =============  =============  ==========  ==========
        onyx        HOME               55229372      104857600      437006           -
        onyx        WORKDIR            61307868              -      519876           -
        gpfs        CENTER                    -   104857600000           1           -
        gold        ARCHIVE_HOME           4136              -          14           -
        viportal    HOME                     32       10485760          13     1000000
        viportal    WORKDIR                   1              -           1           -
        """
        data = utils.parse_show_storage_stdout(stdout.strip(), self.logger)
        assert len(data) == 6

    def test_disk_quota_exceeded(self):
        stdout = """
        ERDC DSRC User Disk Storage Information
                            As of 1255 hours CST 27 November 2023

        System      Location         Disk Usage     Disk Quota  File Usage  File Quota
        ==========  ============  =============  =============  ==========  ==========
        onyx        HOME               55229372      104857600      437006           -
        viportal    HOME               10489999       10485760          13     1000000
        """
        data = utils.parse_show_storage_stdout(stdout.strip(), self.logger)
        assert len(data) == 2 and data[0]["disk_quota_exceed"] == False and data[1]["disk_quota_exceed"] == True
