"""Tests for plugin and/or channel specific tasks"""

import os
import json
import unittest
from types import SimpleNamespace
from plugins.channels import channels
from plugins.filters import BasePluginFilter
from plugins.filters import AvailableOnFilter
from plugins.filters import RequirementsFilter
from plugins.filters import PluginFiltersError
from plugins.filters import filter_wrapper
from plugins.filters import search_wrapper
from plugins.filters import search
import plugins.filters as filters
from plugins import CHANNEL_ROOT_CACHE
from user.user import create_json_file, empty_json_file


class TestChannelRootEnvVarReplacement(unittest.TestCase):

    channel = "Pirates"
    launcher = SimpleNamespace(**{"hpc_variables": {
        "$HOME": "/p/home/jsparrow",
        "$PROJECTS_HOME": "/p/projects",
        "$WORKDIR": "/p/work1/jsparrow"
    }}) # A gimmicked InterfaceLauncher

    def test_basic_home_replacement(self):
        channel_root = "/p/home/jsparrow/test/black-pearl-plugin-channel"
        replaced = "$HOME/test/black-pearl-plugin-channel"
        assert replaced == channels.update_channel_root(self.launcher, channel_root, self.channel)

    def test_basic_projects_home_replacement(self):
        channel_root = "/p/projects/black_pearl/black-pearl-plugin-channel"
        replaced = "$PROJECTS_HOME/black_pearl/black-pearl-plugin-channel"
        assert replaced == channels.update_channel_root(self.launcher, channel_root, self.channel)

    def test_basic_workdir_replacement(self):
        channel_root = "/p/work1/jsparrow/black-pearl-plugin-channel"
        replaced = "$WORKDIR/black-pearl-plugin-channel"
        assert replaced == channels.update_channel_root(self.launcher, channel_root, self.channel)

    def test_multi_replace_issue(self):
        # /p/home/jsparrow should not be replaced
        channel_root = "/p/projects/p/home/jsparrow/black_pearl/black-pearl-plugin-channel"
        replaced = "$PROJECTS_HOME/p/home/jsparrow/black_pearl/black-pearl-plugin-channel"
        assert replaced == channels.update_channel_root(self.launcher, channel_root, self.channel)

    def test_no_replacement(self):
        channel_root = "/tmp/black_pearl/black-pearl-plugin-channel"
        replaced = "/tmp/black_pearl/black-pearl-plugin-channel"
        assert replaced == channels.update_channel_root(self.launcher, channel_root, self.channel)

class TestChannelRootCacheFile(unittest.TestCase):

    cache = "/tmp/channel_root_cache.json"

    @classmethod
    def setUpClass(cls):
        channels.CHANNEL_ROOT_CACHE = cls.cache
        create_json_file(cls.cache)

    @classmethod
    def tearDownClass(cls):
        channels.CHANNEL_ROOT_CACHE = CHANNEL_ROOT_CACHE
        os.remove(cls.cache)

    def setUp(self):
        empty_json_file(self.cache)

    def test_write_cache(self):
        channels.write_channel_root_to_cache("test", "$HOME/test")
        assert channels.get_channel_root_from_cache("test") == "$HOME/test"

    def test_update_existing_channel(self):
        self.test_write_cache()
        channels.write_channel_root_to_cache("test", "$HOME/test2")
        assert channels.get_channel_root_from_cache("test") == "$HOME/test2"

    def test_no_item_in_cache(self):
        assert channels.get_channel_root_from_cache("test") == ""

    def test_remove_item_from_cache(self):
        self.test_write_cache()
        channels.remove_channel_root_from_cache("test")
        assert channels.get_channel_root_from_cache("test") == ""

class TestFilter(unittest.TestCase):

    # Currently the filter methods only require a plugin to have the "title" and "available-on" properties.
    # The plugins.json file only contains the bare minimum data properties for these tests.
    test_dir = os.path.dirname(os.path.abspath(__file__))
    test_plugins_file = os.path.join(test_dir, "plugins/plugins.json")
    test_plugins_data = {}
    test_systems_file = os.path.join(test_dir, "systems/systems.json")
    test_systems_data = {}
    test_groups_file = os.path.join(test_dir, "systems/groups.json")
    test_groups_data = {}

    @classmethod
    def setUpClass(cls):
        # Load test plugins data
        with open(cls.test_plugins_file, 'r') as file:
            cls.test_plugins_data = json.load(file)
        # Load test systems data
        with open(cls.test_systems_file, 'r') as file:
            cls.test_systems_data = json.load(file)
        # Load test groups data
        with open(cls.test_groups_file, 'r') as file:
            cls.test_groups_data = json.load(file)

    def get_filter(self, plugins=None, system=None, node_type=None) -> AvailableOnFilter:
        """Helper method to create an AvailableOnFilter instance for testing"""
        if not system:
            system = "DummySystem"
        if not node_type:
            node_type = "DummyNode"
        if not plugins:
            plugins = self.test_plugins_data
        f = AvailableOnFilter(plugins, system, node_type)
        f.all_systems = self.test_systems_data
        f.all_groups = self.test_groups_data
        return f

    def get_requirements_filter(self, plugins=None, system=None, node_type=None) -> RequirementsFilter:
        """Helper method to create a RequirementsFilter instance for testing"""
        if not system:
            system = "DummySystem"
        if not node_type:
            node_type = "DummyNode"
        if not plugins:
            plugins = self.test_plugins_data
        f = RequirementsFilter(plugins, system, node_type)
        f.all_systems = self.test_systems_data
        f.all_groups = self.test_groups_data
        return f

    def get_base_filter(self, plugins=None, system=None, node_type=None) -> BasePluginFilter:
        """Helper method to create a BasePluginFilter instance for testing"""
        if not system:
            system = "DummySystem"
        if not node_type:
            node_type = "DummyNode"
        if not plugins:
            plugins = self.test_plugins_data
        f = BasePluginFilter(plugins, system, node_type)
        f.all_systems = self.test_systems_data
        f.all_groups = self.test_groups_data
        return f

    def test_get_system(self):
        f = self.get_filter()
        assert f._get_system("Raider") == "Raider"
        assert f._get_system("Raider:cpu:gpu") == "Raider"

    def test_get_group(self):
        f = self.get_filter()
        assert f._get_group("Raider") == None
        assert f._get_group("Raider:cpu:gpu") == None
        assert f._get_group("HPC-Group-x86") == "x86"
        assert f._get_group("HPC-Group-x86:cpu:gpu:another-node-type:abc123") == "x86"

    def test_is_system_match(self):
        f = self.get_filter()
        assert f._is_system_match("Narwhal") == False
        assert f._is_system_match("DummySystem") == True
        # "CustomSystem" exists within dummy system data with an "available_on_map"
        f = self.get_filter(system="CustomSystem")
        assert f._is_system_match("DummySystem") == False
        assert f._is_system_match("Raider") == True
        assert f._is_system_match("AnalyticsGateway") == True

    def test_is_group_match(self):
        f = self.get_filter(system="Raider")
        assert f._is_group_match("All") == True
        assert f._is_group_match("x86") == True
        assert f._is_group_match("ppcle") == False
        # "CustomSystem" exists within dummy system data with an "available_on_map"
        f = self.get_filter(system="CustomSystem")
        assert f._is_group_match("x86") == True
        assert f._is_group_match("AFRL") == False

    def test_is_node_type_match(self):
        f = self.get_filter()
        assert f._is_node_type_match([]) == True
        assert f._is_node_type_match(["n1", "n2", "DummyNode"]) == True
        assert f._is_node_type_match(["n1", "n2"]) == False
        f = self.get_filter(node_type="login")
        assert f._is_node_type_match([]) == False
        assert f._is_node_type_match(["n1", "login", "n2"]) == True
        assert f._is_node_type_match(["login"]) == True

    def test_get_node_types(self):
        f = self.get_filter()
        assert f._get_node_types("Raider") == []
        assert f._get_node_types("Raider:cpu:gpu") == ["cpu", "gpu"]
        assert f._get_node_types("HPC-Group-x86") == []
        assert f._get_node_types("HPC-Group-x86:gpu-2:gpu-4:gpu-8:mla+2:mla+10:mla+1:gpu_x86-64_A100-SXM4_8") == [
            "gpu-2",
            "gpu-4",
            "gpu-8",
            "mla+2",
            "mla+10",
            "mla+1",
            "gpu_x86-64_A100-SXM4_8"
        ]

    def test_get_channel_name(self):
        f = self.get_filter()
        assert f._get_channel_name("a::b") == "a"

    def test_get_plugin_name(self):
        f = self.get_filter()
        assert f._get_plugin_name("a::b") == "b"

    def test_is_system_AG(self):
        f = self.get_filter(system="Public_IT_Analytics_Gateway")
        assert f._is_system_AG() == True
        f = self.get_filter(system="Raider")
        assert f._is_system_AG() == False
        f = self.get_filter(system="CustomSystem")
        assert f._is_system_AG() == False
        f = self.get_filter(system="AnotherSystemThatShouldNotExist")
        assert f._is_system_AG() == False

    def test_is_plugin_AG(self):
        f = self.get_filter()
        assert f._is_plugin_AG("default::CodeServer") == False
        assert f._is_plugin_AG("Public_IT_Analytics_Gateway::CodeServer") == True

    def test_is_plugin_personal(self):
        f = self.get_filter()
        assert f._is_plugin_personal("default::CodeServer") == False
        assert f._is_plugin_personal("notpersonal::CodeServer") == False
        assert f._is_plugin_personal("personal::CodeServer") == True

    def test_get_available_on_map_systems(self):
        f = self.get_filter()
        assert f._get_available_on_map_systems() == []
        f = self.get_filter(system="CustomSystem")
        assert f._get_available_on_map_systems() == ["AnalyticsGateway", "Raider"]

    def test_get_available_on_map_groups(self):
        f = self.get_filter()
        assert f._get_available_on_map_groups() == []
        f = self.get_filter(system="CustomSystem")
        assert f._get_available_on_map_groups() == ["x86"]

    def test_sort_keys_by_plugin_title(self):
        f = self.get_filter()
        assert f._sort_keys_by_plugin_title(["air::air", "air::afsim_login"]) == ["air::afsim_login", "air::air"]

    def test_filter_by_available_on(self):
        # Create a filter with only the original plugins (no requirements plugins)
        original_plugins = {k: v for k, v in self.test_plugins_data.items() if not k.startswith("test::")}

        f = self.get_filter(plugins=original_plugins, system="Raider", node_type="standard")
        expected = ["personal::CodeServer", "kasm-ubuntu::Ubuntu_Desktop", "air::air", "default::TTYD", "default::Jupyter", "default::Jupyter2", "default::Jupyter3"]
        assert sorted(f._filter_by_available_on()) == sorted(expected)

        f = self.get_filter(plugins=original_plugins, system="Raider", node_type="mla")
        expected = ["personal::CodeServer", "default::TTYD", "air::air", "default::Jupyter", "default::Jupyter2", "default::Jupyter3"]
        assert sorted(f._filter_by_available_on()) == sorted(expected)

        f = self.get_filter(plugins=original_plugins, system="Coral", node_type="login")
        expected = ["personal::TTYD"]
        assert sorted(f._filter_by_available_on()) == sorted(expected)

    def test_filter_by_analytics_gateway(self):
        f = self.get_filter(system="Public_IT_Analytics_Gateway", node_type=None)
        expected = ["personal::TTYD", "Public_IT_Analytics_Gateway::Jupyter"]
        assert sorted(f._filter_by_analytics_gateway()) == sorted(expected)

    def test_filter_raises_PluginFiltersError_for_invalid_system(self):
        f = self.get_filter(system="does_not_exixt")
        try:
            f.filter()
        except PluginFiltersError as e:
            if "valid system" in str(e):
                assert True
                return
        assert False

    def test_filter_raises_PluginFiltersError_for_invalid_node_type(self):
        f = self.get_filter(system="Raider", node_type="does_not_exixt")
        try:
            f.filter()
        except PluginFiltersError as e:
            if "valid node type" in str(e):
                assert True
                return
        assert False

    def test_inheritance(self):
        """Test that inheritance is working correctly"""
        base = self.get_base_filter()
        avail = self.get_filter()
        req = self.get_requirements_filter()

        # Test that base class methods are accessible from child classes
        assert hasattr(base, "_sort_keys_by_plugin_title")
        assert hasattr(avail, "_sort_keys_by_plugin_title")
        assert hasattr(req, "_sort_keys_by_plugin_title")

        # Test that the method implementations are the same
        test_keys = ["air::air", "air::afsim_login"]
        assert base._sort_keys_by_plugin_title(test_keys) == avail._sort_keys_by_plugin_title(test_keys)
        assert base._sort_keys_by_plugin_title(test_keys) == req._sort_keys_by_plugin_title(test_keys)

    def test_requirements_filter_mock(self):
        """Test the RequirementsFilter with mocked feature data"""
        req_filter = self.get_requirements_filter(system="DummySystem", node_type="DummyNode")

        # Mock the feature data and system features
        req_filter._feature_data = {
            "architecture": "x86-64",
            "shared": False,
            "webAccess": True,
            "ram": 128,
            "cpu": {
                "cores": 64,
                "coreSpeed": 2.5,
                "vendor": "AMD",
                "model": "EPYC"
            },
            "gpu": {
                "count": 0,
                "ram": 0,
                "vendor": "",
                "model": ""
            }
        }

        req_filter._system_features = {
            "os": {
                "name": "Linux",
                "distribution": "RHEL",
                "version": "8"
            }
        }

        # Test CPU requirements
        assert req_filter._check_cpu_requirements({"cpu": {"vendor": "AMD"}}) == True
        assert req_filter._check_cpu_requirements({"cpu": {"vendor": "Intel"}}) == False
        assert req_filter._check_cpu_requirements({"cpu": {"min": 32}}) == True
        assert req_filter._check_cpu_requirements({"cpu": {"min": 128}}) == False

        # Test RAM requirements
        assert req_filter._check_ram_requirements({"ram_gb": 64}) == True
        assert req_filter._check_ram_requirements({"ram_gb": 256}) == False

        # Test OS requirements
        assert req_filter._check_os_requirements({"os": {"name": "Linux"}}) == True
        assert req_filter._check_os_requirements({"os": {"name": "Windows"}}) == False

        # Test architecture requirements
        assert req_filter._check_arch_requirements({"arch": {"name": "x86-64"}}) == True
        assert req_filter._check_arch_requirements({"arch": {"name": "ARM"}}) == False

        # Test host requirements
        assert req_filter._check_host_requirements({"host": {"shared": False}}) == True
        assert req_filter._check_host_requirements({"host": {"shared": True}}) == False
        assert req_filter._check_host_requirements({"host": {"web_access": True}}) == True
        assert req_filter._check_host_requirements({"host": {"web_access": False}}) == False

        # Test combined requirements
        requirements = {
            "cpu": {"vendor": "AMD", "min": 32},
            "ram_gb": 64,
            "os": {"name": "Linux"},
            "arch": {"name": "x86-64"},
            "host": {"shared": False, "web_access": True}
        }
        assert req_filter._check_all_requirements("test::plugin", requirements) == True

        # Test with non-matching requirements
        requirements = {
            "cpu": {"vendor": "AMD", "min": 128}  # Too many cores required
        }
        assert req_filter._check_all_requirements("test::plugin", requirements) == False

    def test_requirements_filter_with_feature_files(self):
        """Test the RequirementsFilter with actual feature files"""
        # Save the original SYSTEM_FEATURES_DIR
        original_system_features_dir = filters.SYSTEM_FEATURES_DIR
        test_dir = self.test_dir

        try:
            # Mock SYSTEM_FEATURES_DIR to point to our test directory
            filters.SYSTEM_FEATURES_DIR = os.path.join(test_dir, "smartqueue/features")

            # Test with DummySystem
            req_filter = self.get_requirements_filter(system="DummySystem", node_type="DummyNode")

            # Test loading feature data
            feature_data = req_filter._load_feature_data()
            assert feature_data is not None
            assert feature_data.get("architecture") == "x86-64"
            assert feature_data.get("cpu", {}).get("cores") == 64

            # Test with requirements that should match
            requirements = {
                "cpu": {"vendor": "AMD", "min": 32},
                "ram_gb": 64,
                "os": {"name": "Linux"},
                "arch": {"name": "x86-64"}
            }

            # Debug output
            print("DummySystem feature data:", req_filter._feature_data)
            print("DummySystem requirements:", requirements)

            # Check each requirement individually
            print("CPU check:", req_filter._check_cpu_requirements(requirements))
            print("RAM check:", req_filter._check_ram_requirements(requirements))
            print("OS check:", req_filter._check_os_requirements(requirements))
            print("Arch check:", req_filter._check_arch_requirements(requirements))
            print("Host check:", req_filter._check_host_requirements(requirements))
            print("Preinstalled apps check:", req_filter._check_preinstalled_apps(requirements))
            print("Blacklist check:", req_filter._check_blacklist(requirements))

            # Print the result of _check_all_requirements
            all_req_result = req_filter._check_all_requirements("test::plugin", requirements)
            print("All requirements check:", all_req_result)

            # Now that we've confirmed all checks pass, let's assert it
            assert all_req_result == True

            # Test with requirements that should not match
            requirements = {
                "gpu": {"min": 1}  # DummyNode has no GPU
            }
            assert req_filter._check_all_requirements("test::plugin", requirements) == False

            # Test with Raider system
            req_filter = self.get_requirements_filter(system="Raider", node_type="standard")

            # Test loading feature data
            feature_data = req_filter._load_feature_data()
            assert feature_data is not None
            assert feature_data.get("architecture") == "x86-64"
            assert feature_data.get("cpu", {}).get("cores") == 128

            # Test with requirements that should match
            requirements = {
                "cpu": {"vendor": "AMD", "min": 64},
                "ram_gb": 128,
                "os": {"name": "Linux"},
                "arch": {"name": "x86-64"}
            }
            assert req_filter._check_all_requirements("test::plugin", requirements) == True

            # Test with requirements that should not match
            requirements = {
                "gpu": {"min": 1}  # standard node has no GPU
            }
            assert req_filter._check_all_requirements("test::plugin", requirements) == False

            # Test with Raider mla node (has GPU)
            req_filter = self.get_requirements_filter(system="Raider", node_type="mla")

            # Test with GPU requirements that should match
            requirements = {
                "gpu": {"vendor": "NVIDIA", "min": 1}
            }
            assert req_filter._check_all_requirements("test::plugin", requirements) == True

        finally:
            # Restore the original SYSTEM_FEATURES_DIR
            filters.SYSTEM_FEATURES_DIR = original_system_features_dir

    def test_requirements_filter_integration(self):
        """Test the filter method of RequirementsFilter with test plugins"""
        # Save the original SYSTEM_FEATURES_DIR
        original_system_features_dir = filters.SYSTEM_FEATURES_DIR
        test_dir = self.test_dir

        try:
            # Mock SYSTEM_FEATURES_DIR to point to our test directory
            filters.SYSTEM_FEATURES_DIR = os.path.join(test_dir, "smartqueue/features")

            # Test with DummySystem
            plugins = {
                "test::cpu-requirements": self.test_plugins_data["test::cpu-requirements"],
                "test::gpu-requirements": self.test_plugins_data["test::gpu-requirements"],
                "test::os-requirements": self.test_plugins_data["test::os-requirements"],
                "test::arch-requirements": self.test_plugins_data["test::arch-requirements"],
                "test::host-requirements": self.test_plugins_data["test::host-requirements"],
                "test::blacklist-requirements": self.test_plugins_data["test::blacklist-requirements"],
                "test::combined-requirements": self.test_plugins_data["test::combined-requirements"]
            }

            # Test with DummySystem (no GPU)
            req_filter = self.get_requirements_filter(plugins=plugins, system="DummySystem", node_type="DummyNode")
            filtered_plugins = req_filter.filter()

            # CPU, OS, arch, host, and combined requirements should match
            assert "test::cpu-requirements" in filtered_plugins
            assert "test::os-requirements" in filtered_plugins
            assert "test::arch-requirements" in filtered_plugins
            assert "test::host-requirements" in filtered_plugins
            assert "test::combined-requirements" in filtered_plugins

            # GPU requirements should not match
            assert "test::gpu-requirements" not in filtered_plugins

            # Blacklist requirements should not match (DummySystem is blacklisted)
            assert "test::blacklist-requirements" not in filtered_plugins

            # Test with Raider mla node (has GPU)
            req_filter = self.get_requirements_filter(plugins=plugins, system="Raider", node_type="mla")

            # Debug output for Raider mla node
            print("\nRaider mla feature data:", req_filter._feature_data)

            # Get the GPU requirements from the plugin
            gpu_requirements = plugins["test::gpu-requirements"].get("requirements", {})
            print("GPU requirements from plugin:", gpu_requirements)

            # Check if the GPU requirements match
            gpu_match = req_filter._check_all_requirements("test::gpu-requirements", gpu_requirements)
            print("GPU requirements match:", gpu_match)

            # Get the filtered plugins
            filtered_plugins = req_filter.filter()
            print("Filtered plugins:", filtered_plugins)

            # GPU requirements should match
            assert "test::gpu-requirements" in filtered_plugins

        finally:
            # Restore the original SYSTEM_FEATURES_DIR
            filters.SYSTEM_FEATURES_DIR = original_system_features_dir

    def test_filter_wrapper(self):
        """Test the filter_wrapper function"""
        # Create a test environment where we can control the output of the filter methods
        original_filter_method = AvailableOnFilter.filter
        original_req_filter_method = RequirementsFilter.filter
        original_sort_method = BasePluginFilter._sort_keys_by_plugin_title

        try:
            # Mock the filter methods to return known values
            AvailableOnFilter.filter = lambda self: ["avail1", "avail2"]
            RequirementsFilter.filter = lambda self: ["req1", "req2"]
            # Mock the sort method to just return the list as is (avoid accessing self.plugins)
            BasePluginFilter._sort_keys_by_plugin_title = lambda self, keys: sorted(keys)

            # Test that filter_wrapper combines and sorts the results
            result = filter_wrapper("Raider", "standard")

            # The result should be the combined and sorted list
            # Since we're mocking the filter methods, we don't need to check the actual values
            # Just that the function is calling both filter methods and combining the results
            assert len(result) == 4
            assert "avail1" in result
            assert "avail2" in result
            assert "req1" in result
            assert "req2" in result

        finally:
            # Restore the original methods
            AvailableOnFilter.filter = original_filter_method
            RequirementsFilter.filter = original_req_filter_method
            BasePluginFilter._sort_keys_by_plugin_title = original_sort_method

    def test_search_wrapper(self):
        """Test the search_wrapper function"""
        # Create a test environment where we can control the output of the filter_wrapper function
        original_filter_wrapper = filter_wrapper
        original_sort_method = BasePluginFilter._sort_keys_by_plugin_title

        try:
            # Mock the filter_wrapper function to return known values
            def mock_filter_wrapper(system, node_type):
                return ["channel1::plugin1", "channel2::plugin2", "channel3::plugin3"]

            # Replace the real function with our mock
            globals()["filter_wrapper"] = mock_filter_wrapper

            # Mock the sort method to just return the list as is (avoid accessing self.plugins)
            BasePluginFilter._sort_keys_by_plugin_title = lambda self, keys: sorted(keys)

            # Test that search_wrapper filters the results based on the search term
            # We'll need to create a test environment with known plugin data
            test_plugins = {
                "channel1::plugin1": {"title": "Test Plugin 1"},
                "channel2::plugin2": {"title": "Another Plugin"},
                "channel3::plugin3": {"title": "Test Plugin 3"}
            }

            # Save the original PLUGINS_DATA
            import plugins.plugins as plugins_util
            original_plugins_data = plugins_util.PLUGINS_DATA

            try:
                # Replace with our test data
                plugins_util.PLUGINS_DATA = test_plugins

                # Test the search wrapper
                result = search_wrapper("Raider", "standard", "Test")

                # It should return plugins that match the search term
                assert "channel1::plugin1" in result
                assert "channel3::plugin3" in result
                assert "channel2::plugin2" not in result

            finally:
                # Restore the original PLUGINS_DATA
                plugins_util.PLUGINS_DATA = original_plugins_data

        finally:
            # Restore the original filter_wrapper function
            globals()["filter_wrapper"] = original_filter_wrapper
            BasePluginFilter._sort_keys_by_plugin_title = original_sort_method

    def test_fuzzy_search(self):
        """Test the global search function"""
        # Use the global search function instead of an instance method
        assert search("my plugin", ["a", "b", "c"]) == False
        assert search("my plugin", ["a", "myplugin", "c"]) == True
        assert search("Jupyter", ["a", "notjupyter", "c"]) == True
        assert search("Jupyer", ["Code Server", "Jupyter", "Kibana"]) == True
        assert search("Direct Match", ["not a good match", "maybe more direct"]) == False
        assert search("Direct Match", ["not a good match", "maybe more direct", "Direct Match"]) == True
        assert search("code", ["CodeServer"]) == True
        assert search("couch", ["CB+Couchbase+Java"]) == True
