"""Launch policy testing"""

import os
import json
import pathlib
import unittest
from engine.iLauncher import InterfaceLauncher
from engine.launch_policy import LaunchPolicy
from engine import structs
from utils import JOBS_CACHE_FILE

class TestLaunchPolicy(unittest.TestCase):

    # Get file paths relative to this file's known location
    system_file = os.path.join(os.path.dirname(__file__), "../systems.json")
    job_id = "testing123"

    @classmethod
    def setUpClass(cls):
        """Create a dummy iLauncher class from first system in system.json.
        Also create empty job's cache file to avoid FileNotFound error.
        """
        jobs_cache = pathlib.Path(JOBS_CACHE_FILE)
        jobs_cache.parent.mkdir(parents=True, exist_ok=True)
        with jobs_cache.open("w") as f:
            json.dump({}, f)

        with open(cls.system_file, mode='r') as f:
            data = json.load(f)
        hpc_name = ""
        hpc_data = {}
        for name, data in data.get("systems", {}).items():
            hpc_name = name
            hpc_data = data
            break
        cls.launcher = InterfaceLauncher(hpc_name, hpc_data)

    def _setup_test(self, apps, append_cache=False):
        """Creates the launch_policy iterator and updates the launcher's jobs cache with provided apps. This should
        essentially imitate a user already submitting a job and the job's cache being updated through a normal launch
        process. The job's cache will be missing a lot of normal app data, but we're only adding what we need for
        testing.

        Args:
            apps (list): List of apps needed for LaunchPolicy and the launcher object's jobs_cache_info
            append_cache (bool): When true appends to the provided apps to job's cache. Default False
        """
        self.launcher.job_status["system"]["ignored_jobs"] = []
        self.launch_policy = LaunchPolicy(self.launcher, self.job_id, apps)
        cache = {"apps": []}
        for app in apps:
            _app = structs.default_app_cache_dict()
            _app["name"] = app["name"]
            _app["depends_on"] = app["depends_on"]
            cache["apps"].append(_app)
        if append_cache:
            self.launcher.jobs_cache_info[self.job_id]["apps"].extend(cache["apps"])
        else:
            self.launcher.jobs_cache_info[self.job_id] = cache

    def test_single_depends(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": []}
        ]
        expected = ["Jupyter", "Tensorboard"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            app = next(self.launch_policy)
            assert app["name"] == expected[i]
            self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True

    def test_single_depends_multi_app(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": []},
            {"name": "PgAdmin", "cache_index": 2, "depends_on": ["POSTgres"]},
            {"name": "POSTgres", "cache_index": 3, "depends_on": []}
        ]
        expected = ["Jupyter", "POSTgres", "Tensorboard", "PgAdmin"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            app = next(self.launch_policy)
            assert app["name"] == expected[i]
            self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True

    def test_multi_depends_multi_app(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter", "Jupyter", "PgAdmin"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": []},
            {"name": "PgAdmin", "cache_index": 2, "depends_on": ["POSTgres"]},
            {"name": "POSTgres", "cache_index": 3, "depends_on": []}
        ]
        expected = ["Jupyter", "POSTgres", "PgAdmin", "Tensorboard"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            app = next(self.launch_policy)
            assert app["name"] == expected[i]
            self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True

    def test_depends_chain(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": ["PgAdmin"]},
            {"name": "PgAdmin", "cache_index": 2, "depends_on": ["POSTgres"]},
            {"name": "POSTgres", "cache_index": 3, "depends_on": []}
        ]
        expected = ["POSTgres", "PgAdmin", "Jupyter", "Tensorboard"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            app = next(self.launch_policy)
            assert app["name"] == expected[i]
            self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True

    def test_stop_iteration_on_app_success(self):
        apps = [{"name": "Jupyter", "cache_index": 0, "depends_on": []}]
        self._setup_test(apps)
        # Loop 1 more times than necessary to trigger StopIteration
        for i in range(0, len(apps) + 1):
            try:
                app = next(self.launch_policy)
                self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True
            except StopIteration:
                assert i == 1

    def test_no_depends(self):
        apps = [
            {"name": "Jupyter1", "cache_index": 0, "depends_on": []},
            {"name": "Jupyter2", "cache_index": 1, "depends_on": []},
            {"name": "Jupyter3", "cache_index": 2, "depends_on": []},
        ]
        expected = ["Jupyter1", "Jupyter2", "Jupyter3"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            app = next(self.launch_policy)
            #self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True

    def test_app_fail_with_depends_chain(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": ["PgAdmin"]},
            {"name": "PgAdmin", "cache_index": 2, "depends_on": ["POSTgres"]},
            {"name": "POSTgres", "cache_index": 3, "depends_on": []}
        ]
        expected = ["POSTgres", "PgAdmin", "Jupyter"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            try:
                app = next(self.launch_policy)
                assert app["name"] == expected[i]
                if app["name"] == "Jupyter":
                    self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["tunnel_status"] = 255
                else:
                    self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True
            except StopIteration:
                assert i == 3

    def test_app_fail_and_no_depends_app_still_launch(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": []},
            {"name": "PgAdmin", "cache_index": 2, "depends_on": ["POSTgres"]},
            {"name": "POSTgres", "cache_index": 3, "depends_on": []}
        ]
        expected = ["Jupyter", "POSTgres", "PgAdmin"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            try:
                app = next(self.launch_policy)
                assert app["name"] == expected[i]
                if app["name"] == "Jupyter":
                    self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["tunnel_status"] = -1
                else:
                    self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True
            except StopIteration:
                assert i == 3

    def test_apps_stopped_by_user(self):
        apps = [
            {"name": "Tensorboard", "cache_index": 0, "depends_on": ["Jupyter", "PgAdmin"]},
            {"name": "Jupyter", "cache_index": 1, "depends_on": []},
            {"name": "PgAdmin", "cache_index": 2, "depends_on": []},
        ]
        expected = ["Jupyter"]
        self._setup_test(apps)
        for i in range(0, len(apps)):
            try:
                app = next(self.launch_policy)
                assert app["name"] == expected[i]
                if i == 0:
                    self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["remote_server_running"] = True
                    self.launcher.jobs_cache_info[self.job_id]["apps"][0]["stopped"] = True
                    self.launcher.jobs_cache_info[self.job_id]["apps"][1]["stopped"] = True
                    self.launcher.jobs_cache_info[self.job_id]["apps"][2]["stopped"] = True
            except StopIteration:
                assert i == 1
                break

    def test_apps_added_to_existing_job(self):
        apps = [{"name": "Jupyter", "cache_index": 0, "depends_on": []}]
        self._setup_test(apps)
        apps = [{"name": "PgAdmin", "cache_index": 1, "depends_on": []}]
        self._setup_test(apps, append_cache=True)
        for i in range(0, len(apps)):
            app = next(self.launch_policy)
            assert app["name"] == "PgAdmin"

    def test_job_deleted(self):
        apps = [{"name": "Jupyter", "cache_index": 0, "depends_on": []}]
        self._setup_test(apps)
        # delete job from cache to mimick user deleted job
        self.launcher.jobs_cache_info = {}
        self.launcher.job_status["system"]["ignored_jobs"].append(self.job_id)
        for i in range(0, len(apps)):
            try:
                app = next(self.launch_policy)
            except StopIteration:
                assert i == 0

    def test_job_deleted_and_still_in_cache(self):
        apps = [{"name": "Jupyter", "cache_index": 0, "depends_on": []}]
        self._setup_test(apps)
        self.launcher.job_status["system"]["ignored_jobs"].append(self.job_id)
        for i in range(0, len(apps)):
            try:
                app = next(self.launch_policy)
            except StopIteration:
                assert i == 0

    def test_apps_remote_server_never_started_with_depends(self):
        apps = [
            {"name": "CodeServer", "cache_index": 0, "depends_on": []},
            {"name": "Tensorboard", "cache_index": 1, "depends_on": ["Jupyter"]},
            {"name": "Jupyter", "cache_index": 2, "depends_on": []}
        ]
        self._setup_test(apps)
        expected = ["CodeServer", "Jupyter"]
        for i in range(0, len(apps)):
            try:
                app = next(self.launch_policy)
                assert app["name"] == expected[i]
                if app["name"] == "Jupyter":
                    self.launcher.jobs_cache_info[self.job_id]["apps"][app["cache_index"]]["timed_out"] = True
            except StopIteration:
                assert i == 2
                break
